;;; -*- Mode:Common-Lisp; Package:USER; Fonts:(CPTFONT HL12B HL12I CPTFONT CPTFONTB); Base:10 -*-


4(DEFUN READ-MACPAINT-DOCUMENT (filename &key (height 720) (width 576)*
				4  (rotate 90) (show-bitmap-p t)*
				4  (arrow-key-displacement 50) (verbose t))*
   "2   Read a MacPaint document (Macintosh file type 'PNTG') from FILENAME, convert
it into an Explorer bitmap, rotate the bitmap ROTATE degrees (0, 90, 180, or 270),
show the bitmap if SHOW-BITMAP-P is true, and finally return the rotated bitmap.
   HEIGHT and WIDTH default to the standard MacPaint document size (i.e., a full
ImageWriter printed page).
   If SHOW-BITMAP-P is true, then ARROW-KEY-DISPLACEMENT specifies the number 
of pixels the bitmap being shown will be moved each time an arrow (i.e., cursor)
key is pressed.
   If VERBOSE is true, then progress messages are printed out during processing.*" 
  (declare (values rotated-bitmap))
 
  (let* ((bitmap            nil)
	 (rotated-bitmap    nil)  ;1 BITMAP rotated ROTATE degrees*
	 (byte-vector       nil)  ;1 expanded contents of STREAM-DATA as 8-bit bytes*
	 (swapped-array     (let ((array (make-array 256 :element-type '(mod 256)))
				  (byte-r 0))
			      (dotimes (i 256 array)
				(dotimes (bit 8)
				  (setf byte-r (dpb (ldb (byte 1 bit) i)
						    (byte 1 (- 7 bit)) byte-r)))
				(setf (aref array i) byte-r))) )
	 (stream-length     0)
	 (stream-data       nil)  ;1 holds entire file as and array of (unsigned-byte 8)*
	 (version           0)    ;1 MacPaint file version number*
	 (PatArray          nil)  ;1 38 x 8 array of (unsigned-byte 8) patterns*
	 (Future            nil)  ;1 204 byte vector of (signed-byte 8)*
	 (stream-index      0)
	 (byte-index        0)
	 (count-byte        0)
	1  *);1;let bindings*
    (labels (4(LEGAL-WIDTH-P (value)*
	      "2Returns true of value is >0 and a multiple of 32*"
	      (and (plusp value) (zerop (mod value 32))))
           
	     4(REPEAT-COUNT-P(count-byte)*
	      "2Returns true of COUNT-BYTE is a repeat count*"
	      (logtest #x80 count-byte))

	     4(COUNT* 4(count-byte)*
	      "2Returns the count encoded in COUNT-BYTE*"
	      (if (repeat-count-p count-byte)
		  ;1; then this is a repeat count for next byte*
		  (1+ (- #x100 count-byte))
		  ;1; else this is a length count of unique bytes to follow*
		  (1+ count-byte)))
	    
	     4(READ-STREAM-DATA-BYTE ()*
	      "2Increment STREAM-INDEX.  Returns STREAM-DATA byte at STREAM-INDEX
3               *before it was incremented.*"
	      (prog1 (aref stream-data stream-index)
	 	     (incf stream-index)))

	     4(WRITE-BYTE-VECTOR-BYTE (byte)*
	      "2Write BYTE to BYTE-VECTOR at BYTE-INDEX then increment BYTE-INDEX.*"
	      (setf (aref byte-vector byte-index) byte)
	      (incf byte-index))

	     4(BIT-SWAP (8-bit-value)*
	      "2Reverse an 8 bit thing*"
	      (aref swapped-array 8-bit-value))
	1       *);1;labels bindings*

      ;1; validate arguments*
      (check-type filename (or string pathname))
      (check-type height   (integer 0 *))
      (check-type width    (satisfies legal-width-p) "3a positive multiple of 32*")
      (check-type rotate   (member 0 90 180 270))

      ;1; read MacPaint file into memory*
      (setf filename (merge-pathnames filename (make-pathname :host sys:local-host 
							      :type :UNSPECIFIC)))
      (when verbose
	(format t "3~%Reading MacPaint document \"~A\"...*" filename))
      (with-open-file (stream filename :direction :input
			      :element-type '(unsigned-byte 8))
	 (setf stream-length (file-length stream :element-type '(unsigned-byte 8)))
	 (when (<= stream-length 512)
	   ;1; then there is nothing to process, so return*
	   (warn "3~%~A stream length is only ~D bytes long.  Nothing to process.*"
		 (namestring stream) stream-length)
	   (return-from read-macpaint-document nil 0 0))
	 
	 ;1; read version (LONGINT)*
	 (dotimes ( i 4)
	   (setf version (+ (* version 8) (read-byte stream))))
	 (when (/= version 0)
	   (warn "3~%NOTE:  This is a version ~D MacPaint file.~
                  ~%       It will be converted as a version 0 file.*" version))

	 ;1; read PatArray (Array [1..38] of Pattern (unsigned-byte 64))*
	 (setf PatArray (make-array (list 38 8) :element-type '(unsigned-byte 8)))
	 (dotimes (i 38)
	   (dotimes (j 8)			       ;1 byte-reverse 8-byte patterns*
	     (setf (aref PatArray i (- 7 j)) (read-byte stream))))

	 ;1; read Future (Array [1..204] of (signed-byte 8)) then byte swap each byte*
	 (setf Future (make-array 204 :element-type '(signed-byte 8)))
	 (send stream :string-in t Future)
	 (dotimes (i 204)
	   (setf (aref Future i) (aref Future i)))

	 ;;1 read the entire remaining file in one gulp*
	 (setf stream-data (make-array (- stream-length (file-position stream))
					:element-type '(unsigned-byte 8)))
	 (send stream :string-in t stream-data)
	 );1;with-open-file*
      (when verbose (3format t *"done."3)*)

      ;1;create bitmap and byte-vector*
      (when verbose (format t "3~%Expanding compressed MacPaint format...*"))
      (setf bitmap      (make-array (list height width) :element-type 'bit))
      (setf byte-vector (make-array (ceiling (* height width) 8)
				    :element-type '(unsigned-byte 8)
				    :displaced-to bitmap))
      ;1; From this point, there should be exactly enough bytes in STREAM-ARRAY such*
      ;1; that, when expanded, they will exactly fill BYTE-VECTOR.*

      ;1; Expand compressed MacPaint format*
      (loop
	(when (>= byte-index (length byte-vector))
	  (return stream-index))
	(setf count-byte (read-stream-data-byte))
	(if (repeat-count-p count-byte)
	    ;1; then this count byte is a repeat count for the next STREAM-DATA byte*
	    (let ((data-byte (bit-swap (read-stream-data-byte))))
	      (dotimes (i (count count-byte))
		(write-byte-vector-byte data-byte)))
	    ;1; else this count byte is a field length for the next STREAM-DATA field*
	    (dotimes (i (count count-byte))
	      (write-byte-vector-byte (bit-swap (read-stream-data-byte)))))
	);1;loop*
      (when verbose (format t "3done.*"))
      
      ;1; Create ROTATED-BITMAP*
      (when (and verbose (not (zerop rotate)))
	(format t "3~%Rotating bitmap ~D degrees...*" rotate))
      (case rotate
	((90 270)
	1  *;1; case of ROTATED-BITMAP = BITMAP with dimensions exchanged*
	 (setf rotated-bitmap			       ;12nd dimension must by x32*
	       (make-array
		 (list (array-dimension bitmap 1)
		       (* (truncate (+ (array-dimension bitmap 0) 31) 32) 32))
		 :element-type 'bit)))
	(180
	 ;1; case of ROTATED-BITMAP = BITMAP*
	 (setf rotated-bitmap
	       (make-array (array-dimensions bitmap) :element-type 'bit)))
	);1;case*

      ;1; Rotate BITMAP*
      (case rotate
	(0   (setf rotated-bitmap bitmap))
	(90  (tv:rotate-90 bitmap  rotated-bitmap))
	(180 (tv:rotate-180 bitmap rotated-bitmap))
	(270 (tv:rotate-270 bitmap rotated-bitmap)))
      (when (and verbose (not (zerop rotate))) (format t "3done*"))
	      
      ;1; Show bitmap, if requested*
      (when show-bitmap-p
	(when verbose (format t "3~%Showing bitmap (use arrow keys to move it)...*"))
	(unwind-protect
	    (tv:show-bit-array rotated-bitmap :window *terminal-io*
			       :increment arrow-key-displacement)
	  (send *terminal-io* :refresh)
	  (when verbose (format t "3~&done.*"))))
      );1;labels*

    (when (and verbose *print-array*)
      (format t "3~%Forcing *PRINT-ARRAY* to NIL to suppress printing ~
                   bitmap contents.*"))
    (setf *print-array* nil)
    rotated-bitmap
    );1;let**
  );1;read-macpaint-document
